<#
    .SYNOPSIS
    Mounts a VHD file to a directory.

    .DESCRIPTION
    Intended for use in incident response, you can take a series of .vhd or .vhdx files gathered using KAPE, mount each to a directory, and then easily view their contents. Script assumes file names follow this format: SourceComputerName-triage.vhdx, SourceComputerName-triage.vhd, SourceComputerName.vhdx, or SourceComputerName.vhd. Files beginning with a date and time stamp in the format YYYY-MM-DDTHHMMSS are also supported if the -StripDateTime switch is used.
    
    Mounting requires administrator permissions, so the PowerShell terminal must run as administrator. 
    
    .PARAMETER VhdDirectory
    Specifies the directory where the unmounted .vhd or .vhdx files are stored. Files should be in the root of this directory. Script does not recurse into subfolders.

    .PARAMETER MountDirectory
    Specifies the directory where mount points for each mounted image will be created. Within the mount directory, a new directory (mount point) will be created. Each directory will be named after the computer from which the VHDX image was captured, based on the filename of the VHDX file. The directory must already exist.
    
    .PARAMETER StripDateTime
    An optional switch that specifies whether to remove dates in the format YYYY-MM-DDTHHMMSS from the beginning of the disk image file name when determining the associated computer name. The default is false. If the switch is specified, the script will remove the date and an underscore from the beginning of the file name before then calculating the computer name.
    
    .EXAMPLE
    PS> Mount-TriageVhdx.ps1 -VhdDirectory "D:\VHDXTriageImagesFolder\" -MountDirectory "D:\MountedImagesFolder\"
    This example mounts disk images located in D:\VHDXTriageImagesFolder\ to D:\MountedImagesFolder\.

    .EXAMPLE
    PS> Mount-TriageVhdx.ps1 -VhdDirectory "D:\VHDXTriageImagesFolder\" -MountDirectory "D:\MountedImagesFolder\" -StripDateTime
    This example mounts disk images located in D:\VHDXTriageImagesFolder\ to D:\MountedImagesFolder\ removing the date and time stamp from the beginning of the file name before calculating the computer name.

    .NOTES
    Written by Steve Anson and Dr. Sorot Panichprecha. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.
#>

param (
    [Parameter(Position=0,mandatory=$true)]
    $VhdDirectory,
    [Parameter(Position=1,mandatory=$true)]
    $MountDirectory,
    [Parameter(mandatory=$false)]
    [switch]$StripDateTime
)

# Check if the current PowerShell session is running as administrator
try {
    $currentPrincipal = [Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()
    if (-NOT $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
        throw 
    }
}
catch {
    Write-Host -ForegroundColor Red "ERROR: This script must be run as Administrator."
    exit 1
}

if ([string]::IsNullOrWhitespace($VhdDirectory)) {
    Write-Host "ERROR: Please specify -VhdDirectory"
    Exit 1
}

# Check if the mount directory exists. If not, exit the program.
if (-Not (Test-Path -Path $MountDirectory)) {
    Write-Host "ERROR: $MountDirectory does not exist"
    Exit 1
}


# Provide info to user and confirm they want to proceed
Write-Host "This script will mount all .vhd and .vhdx files in $VhdDirectory to $MountDirectory"
Write-Host -ForegroundColor Yellow "NOTICE: This script is intended to be used on COPIES of uncompressed evidence files, not on original evidence." 
Write-Host "VHD and VHDX disk images must be initialized to work. `nUnitialized disk images will be mounted read-write and initialized. This will cause the disk image's hash value to change."
Write-Host "Once intialized, the disk image will be mounted read-only for your convenience. This is not a guarantee that the disk image will not be modified."
Write-Host -ForegroundColor Red "Do not use this script on original evidence."

while ($true) {
    $response = Read-Host "Do you want to continue? (Y/N)"
    if ($response -imatch "^(y|yes)$") {
        break
    }
    elseif ($response -imatch "^(N|No)$") {
        Write-Host "Exiting..."
        exit
    }
    else {
        Write-Host "Invalid response. Please enter Y or N."
    }
}

# Validate filename formats

    if ($StripDateTime) {
        Get-ChildItem -Path $VhdDirectory -Include *.vhd,*.vhdx -Recurse | Where-Object { -not $_.PSIsContainer } | ForEach-Object {
            $VhdFileName = $_.Name
            $dateTimeString = $VhdFileName.Substring(0, 17)
            $format = "yyyy-MM-ddTHHmmss"
            $date = New-Object DateTime
            # Check if the first 17 characters of the file name are a properly formatted date and time stamp
            if (-not [DateTime]::TryParseExact($dateTimeString, $format, [Globalization.CultureInfo]::InvariantCulture, [Globalization.DateTimeStyles]::None, [ref]$date)) {
                Write-Host -ForegroundColor Red "ERROR: Unexpected file name format for $VhdFileName. `n-StripDateTime expects file name to begin with a properly formatted date and time stamp in the format YYYY-MM-DDTHHMMSS."
                Write-Host -ForegroundColor Red "No files have been processed. Please correct file names and rerun the script."
                exit 1
            }
        }
    }

# Warn that only .vhd and .vhdx files are supported. All other files will be ignored.
Get-ChildItem -Path $VhdDirectory -Exclude *.vhd,*.vhdx -Recurse | Where-Object { -not $_.PSIsContainer } | ForEach-Object {
    Write-Host -ForegroundColor Yellow "WARNING: Only .vhd and .vhdx files are supported. $_ will be ignored."
}

# Determine name of the computer from which the image was captured, based on file name.

Get-ChildItem -Path $VhdDirectory -Include *.vhd,*.vhdx -Recurse | Where-Object { -not $_.PSIsContainer } | ForEach-Object {
    $VhdFileName = $_.Name
    $CurrentImagePath = $_.FullName
    
    if ($StripDateTime) {
        # Extract the first 17 characters of the file name
        $dateTimeString = $VhdFileName.Substring(0, 17)
        $format = "yyyy-MM-ddTHHmmss"
        $date = New-Object DateTime
        # Check if the first 17 characters of the file name are a properly formatted date and time stamp
        if (-not [DateTime]::TryParseExact($dateTimeString, $format, [Globalization.CultureInfo]::InvariantCulture, [Globalization.DateTimeStyles]::None, [ref]$date)) {
            Write-Host -ForegroundColor Red "ERROR: Unexpected file name format for $VhdFileName. The file name does not begin with a properly formatted date and time stamp."
            Write-Host -ForegroundColor Red "Please ensure that all file names begin with a properly formatted date and time stamp in the format YYYY-MM-DDTHHMMSS."
            Write-Host -ForegroundColor Red "No files have been processed. Please correct file names and rerun the script."
            exit 1        
        }
        # Cut the first 18 characters from the file name for further processing to find the appropriate computer name
        $VhdFileName = ($VhdFileName).substring(18)
    }

    if ($VhdFileName -like "*-triage.vhd") {
        $ComputerName = ($VhdFileName).substring(0, $VhdFileName.Length - 11)
    }
    elseif ($VhdFileName -like "*-triage.vhdx") {
        $ComputerName = ($VhdFileName).substring(0, $VhdFileName.Length - 12)
    }
    elseif ($VhdFileName -like "*.vhd") {
        $ComputerName = ($VhdFileName).substring(0, $VhdFileName.Length - 4)
    }
    elseif ($VhdFileName -like "*.vhdx") {
        $ComputerName = ($VhdFileName).substring(0, $VhdFileName.Length - 5)
    }
    else {
        Write-Host -ForegroundColor Red "ERROR: Unexpected file name format for $VhdFileName."
        Write-Host -ForegroundColor Red "This file will be ignored."
        continue
    }

    # Check if the computer name directory exists. If not, create it.
    if (-Not (Test-Path -Path "$MountDirectory/$ComputerName")) {
        [void](New-Item -Path "$MountDirectory/$ComputerName" -ItemType Directory)
    }
    

    # Mount each VHD or VHDX file in the VHD directory to a subfolder named after the computer from which the image was captured, based on file name.
    # Mount the VHDX file to the computer name directory.
    Write-Host "Processing $_..."
    try {
            $DiskImage = Mount-DiskImage -ImagePath $CurrentImagePath -NoDriveLetter -Access ReadOnly -ErrorAction Stop -Passthru
    }
    catch {
        Write-Host -ForegroundColor Red "ERROR: ***The $ComputerName VHD was not able to be mounted successfully.***"
        Write-Host -ForegroundColor Red "The error message returned was $_"
        Write-Host -ForegroundColor Red "No data was processed from the $ComputerName image."
        continue
    }
    try{
        if ($DiskImage) {
            $disk = $DiskImage | Get-Disk -ErrorAction Stop
        }
    }
    # If an image has never been mounted, it is not initialized and mounting read only will cause get-disk to fail. In that case, briefly mount it read-write to initialize it, then dismount it and mount it read-only.
    catch {
        Write-Host -ForegroundColor Yellow "WARNING: The $ComputerName image was not previously initialized. Remounting the image read-write to initialize it."
        Dismount-DiskImage -ImagePath $CurrentImagePath | Out-Null
        Start-Sleep -Seconds .5
        Mount-DiskImage -ImagePath $CurrentImagePath -NoDriveLetter -ErrorAction Stop -Passthru | Out-Null
        Start-Sleep -Seconds .5
        Write-Host -ForegroundColor Yellow "The image file's hash value has likely changed as a result, but the hashes of the evidence files it contains have not."
        Dismount-DiskImage -ImagePath $CurrentImagePath | Out-Null
        Start-Sleep -Seconds .5
        $DiskImage = Mount-DiskImage -ImagePath $CurrentImagePath -NoDriveLetter -Access ReadOnly -ErrorAction Stop -Passthru
        # Attempt to get the disk again after re-mounting the image
        try {
            if ($DiskImage) {
                $disk = $DiskImage | Get-Disk -ErrorAction Stop
            }
        }
        catch {
            Write-Host -ForegroundColor Red "ERROR: ***The $ComputerName VHD was not able to be mounted successfully after initialization.***"
            Write-Host -ForegroundColor Red "The error message returned was $_"
            Write-Host -ForegroundColor Red "No data was processed from the $ComputerName image."
        }
        Write-Host -ForegroundColor Yellow "The disk image has been intialized and remounted read only."
    }
    try {
            if ($disk) {
                Write-Host "Mounting $_ at $MountDirectory$ComputerName"
                $disk | Get-Partition -ErrorAction Stop | Where-Object { ($_ | Get-Volume) -ne $Null } | Add-PartitionAccessPath -ErrorAction Stop -AccessPath "$MountDirectory/$ComputerName"
            }
    }
    catch {
        Write-Host -ForegroundColor Red "ERROR: ***The $ComputerName VHD was not able to be mounted successfully.***"
        Write-Host -ForegroundColor Red "The error message returned was $_"
        Write-Host -ForegroundColor Red "No data was processed from the $ComputerName image."
        continue
    }

    # Pause for half a second to allow the disk to mount before moving on to the next one.
    Start-Sleep -Seconds .5
}



